#include "nanovoid.h"
#include "cxxopts.hpp"

valueType** init_zero_matrix(uint Nx, uint Ny) {
  valueType** mtx = new valueType*[Nx];
  for (uint i = 0; i < Nx; ++ i){
    valueType* row = new valueType[Ny];
    for (uint j = 0; j < Ny; ++ j)
      row[j] = 0.0;
    mtx[i] = row;
  }
  return mtx;
}

valueType** micro_ch_pre(uint Nx, uint Ny, valueType c0, uint iflag) {
  //uint NxNy = Nx*Ny;
  //valueType noise = 0.02;

  valueType** con = init_zero_matrix(Nx, Ny);
  //valueType threshold = 0.2;
  
  if (iflag == 1){
    /*con = np.zeros((Nx,Ny))
    for i in range(Nx):
      for j in range(Ny):
    con[i,j] = c0 + noise*(threshold-np.random.rand());
    */
    printf("in micro_ch_pre: iflag=%u, not implemented!\n", iflag);
    fflush(stdout);
    assert(false);
  }else if(iflag==2){
    /*
        con = np.zeros((NxNy,1))
        for i in range(Nx):
            for j in range(Ny):
                ii =(i)*Nx+j
                    con[ii] =c0 + noise*(0.5-np.random.rand());
    */
    printf("in micro_ch_pre: iflag=%u, not implemented!\n", iflag);
    fflush(stdout);
    assert(false);
  }else if(iflag == 3){
    //con[int(Nx/4):int(Nx/2),:] = c0;
    for (uint i = Nx/4; i < Nx/2; ++ i)
      for (uint j = 0; j < Ny; ++ j)
        con[i][j] = c0;
  }else if(iflag==4){
    /*
        con = np.zeros((Nx,Ny))
        d = 80
        rr, cc = circle(int(Nx/2), int(Ny/2), d)
        cord = (rr,cc)
          con[cord] = c0;
    */
    printf("in micro_ch_pre: ifflag=%u, not implemented!\n", iflag);
    fflush(stdout);
    assert(false);
  }

  return con;
}


void print_img_in_csv(valueType** img, const char* filename, uint Nx, uint Ny) {
  FILE* oup = fopen(filename, "w");
  fprintf(oup, "%u,%u\n", Nx, Ny);
  for (uint i = 0; i < Nx; ++ i) {
    fprintf(oup, "%lf", img[i][0]);
    for (uint j = 1; j < Ny; ++ j) {
      fprintf(oup, ",%lf", img[i][j]);
    }
    fputc('\n', oup);
  } 
  fclose(oup);
}


class Args {
public:
  uint nsteps;
  string output;
  string bucket_output;
  uint lshL, lshK;
  double lshr;
};

Args* parse_args(int argc, const char* argv[]) {
  try
  {
    Args* args = new Args;
    
    cxxopts::Options options(argv[0], " - test forward simulation of spinodal decomposition");
    options
      .positional_help("[optional args]")
      .show_positional_help();

    options
      .set_width(70)
      .set_tab_expansion()
      .allow_unrecognised_options()
      .add_options()
      ("s,nsteps", "Number of steps of simulation (default=100)", cxxopts::value<int>(), "N")
      ("o,output", "Output file (default=spinodal.out)", cxxopts::value<std::string>(), "FILE")
      ("lshK", "K for LSH (default=1)", cxxopts::value<int>(), "INT")
      ("lshL", "L for LSH (default=1)", cxxopts::value<int>(), "INT")
      ("lshr", "r for LSH (default=1e-4)", cxxopts::value<float>(), "FLOAT")
      ("bucket_output", "Output file of the bucket information (default=bucket.out)", cxxopts::value<std::string>(), "FILE")
      ("h,help", "Print help")
    #ifdef CXXOPTS_USE_UNICODE
      ("unicode", u8"A help option with non-ascii: à. Here the size of the"
        " string should be correct")
    #endif
    ;

    auto result = options.parse(argc, argv);

    if (result.count("help"))
    {
      std::cout << options.help({"", "Group"}) << std::endl;
      exit(0);
    }

    std::cout << "[Parse Args]" << std::endl;
    
    if (result.count("nsteps"))
    {
      std::cout << "  nsteps = " << result["nsteps"].as<int>() << std::endl;
      args->nsteps = (uint)result["nsteps"].as<int>();
    }else{
      args->nsteps = 100;
    }
    
    if (result.count("output"))
    {
      std::cout << "  output = " << result["output"].as<std::string>()
        << std::endl;
      args->output = result["output"].as<std::string>();
    }else{
      args->output = "spinodal.out";
    }
    
    if (result.count("bucket_output"))
    {
      std::cout << "  bucket_output = " << result["bucket_output"].as<std::string>()
        << std::endl;
      args->bucket_output = result["bucket_output"].as<std::string>();
    }else{
      args->bucket_output = "bucket.out";
    }

    if (result.count("lshK"))
    {
      std::cout << "  lshK = " << result["lshK"].as<int>()
        << std::endl;
      args->lshK = (uint)result["lshK"].as<int>();
    }else{
      args->lshK = 1;
    }

    if (result.count("lshL"))
    {
      std::cout << "  lshL = " << result["lshL"].as<int>()
        << std::endl;
      args->lshL = (uint)result["lshL"].as<int>();
    }else{
      args->lshL = 1;
    }

    if (result.count("lshr"))
    {
      std::cout << "  lshr = " << result["lshr"].as<float>()
        << std::endl;
      args->lshr = (double)result["lshr"].as<float>();
    }else{
      args->lshr = 1e-4;
    }
    
    auto arguments = result.arguments();
    std::cout << "  Saw " << arguments.size() << " arguments" << std::endl;

    std::cout << "[End of Parse Args]" << std::endl;

    return args;
  }
  catch (const cxxopts::OptionException& e)
  {
    std::cout << "error parsing options: " << e.what() << std::endl;
    exit(1);
  }
}


int main(int argc, const char* argv[]) {
  Args* args = parse_args(argc, argv);

  // def parameters
  uint Nx = 512; //1024;
  uint Ny = 512; //1024;

  uint K = args->lshK;
  uint L = args->lshL;
  valueType lsh_r = args->lshr;

  valueType h = 1.0;

  uint nsteps = args->nsteps;
  valueType dtime = 1e-2;
  valueType ttime = 0.0;

  valueType A = 1.0;
  valueType c0 = 0.5;
  valueType mobility = 10.0;
  valueType kappa = 0.5;

  uint iflag = 3;

  // form initial matrix
  valueType** con = micro_ch_pre(Nx, Ny, c0, iflag);

  // form OneStep class
  assert(Nx == Ny);

  printf("before construct one_step\n");
  SpinodalDecompOneStep one_step(lsh_r, Nx, K, L, A, kappa, mobility, dtime, h);
  printf("before encode_from_img\n");
  fflush(stdout);
  one_step.encode_from_img(con);
  printf("finish encode_from_img\n");
  fflush(stdout);
  
  //  one_step.hash_t.print_hash_table(one_step.inv, args->bucket_output.c_str());

  for (uint step = 0; step < nsteps; ++ step) {
    printf("step=%u, ttime=%lf\n", step, ttime);
    one_step.next();
    ttime += dtime;

    if (step % 10 == 0) {
      one_step.inv.check_from_dfslist(Nx*Ny);
    }
  }

  valueType** end_result = one_step.decode_to_img();

  print_img_in_csv(end_result, args->output.c_str(), Nx, Ny);

  // print all buckets
  one_step.print_PNBuckets_to_file(args->bucket_output.c_str());

  delete args;
  
  printf("Have a nice day!\n");
  return 0;
}
